import XCTest
import CoreData
@testable import death_app_Watch_App

final class PredictionHistoryTests: XCTestCase {
    
    var databaseService: DatabaseService!
    var predictionHistoryManager: PredictionHistoryManager!
    var testContext: NSManagedObjectContext!
    
    override func setUpWithError() throws {
        try super.setUpWithError()
        
        // Create in-memory Core Data stack for testing
        let persistentContainer = NSPersistentContainer(name: "LifeClock")
        let description = NSPersistentStoreDescription()
        description.type = NSInMemoryStoreType
        persistentContainer.persistentStoreDescriptions = [description]
        
        persistentContainer.loadPersistentStores { _, error in
            if let error = error {
                fatalError("Failed to load test store: \(error)")
            }
        }
        
        testContext = persistentContainer.viewContext
        databaseService = DatabaseService(context: testContext)
        predictionHistoryManager = PredictionHistoryManager(databaseService: databaseService)
    }
    
    override func tearDownWithError() throws {
        testContext = nil
        databaseService = nil
        predictionHistoryManager = nil
        try super.tearDownWithError()
    }
    
    // MARK: - Database Operations Tests
    
    func testSavePrediction() async throws {
        let prediction = PredictionResult(
            date: Date(),
            lifeExpectancy: 78.5,
            riskFactors: ["smoking": 0.8, "exercise": 0.3],
            confidence: 0.85
        )
        
        await databaseService.savePrediction(prediction)
        
        let savedPredictions = await databaseService.getAllPredictions()
        XCTAssertEqual(savedPredictions.count, 1)
        XCTAssertEqual(savedPredictions.first?.lifeExpectancy, 78.5)
    }
    
    func testGetPredictionsByDateRange() async throws {
        let now = Date()
        let yesterday = Calendar.current.date(byAdding: .day, value: -1, to: now)!
        let tomorrow = Calendar.current.date(byAdding: .day, value: 1, to: now)!
        
        // Save predictions on different dates
        let prediction1 = PredictionResult(date: yesterday, lifeExpectancy: 75.0, riskFactors: [:], confidence: 0.8)
        let prediction2 = PredictionResult(date: now, lifeExpectancy: 78.0, riskFactors: [:], confidence: 0.8)
        let prediction3 = PredictionResult(date: tomorrow, lifeExpectancy: 80.0, riskFactors: [:], confidence: 0.8)
        
        await databaseService.savePrediction(prediction1)
        await databaseService.savePrediction(prediction2)
        await databaseService.savePrediction(prediction3)
        
        let predictions = await databaseService.getPredictions(from: yesterday, to: now)
        XCTAssertEqual(predictions.count, 2)
    }
    
    func testDeleteOldPredictions() async throws {
        let now = Date()
        let oldDate = Calendar.current.date(byAdding: .day, value: -100, to: now)!
        let recentDate = Calendar.current.date(byAdding: .day, value: -10, to: now)!
        
        let oldPrediction = PredictionResult(date: oldDate, lifeExpectancy: 75.0, riskFactors: [:], confidence: 0.8)
        let recentPrediction = PredictionResult(date: recentDate, lifeExpectancy: 78.0, riskFactors: [:], confidence: 0.8)
        
        await databaseService.savePrediction(oldPrediction)
        await databaseService.savePrediction(recentPrediction)
        
        let cutoffDate = Calendar.current.date(byAdding: .day, value: -30, to: now)!
        await databaseService.deleteOldPredictions(before: cutoffDate)
        
        let remainingPredictions = await databaseService.getAllPredictions()
        XCTAssertEqual(remainingPredictions.count, 1)
        XCTAssertEqual(remainingPredictions.first?.date.timeIntervalSince1970, recentDate.timeIntervalSince1970, accuracy: 1.0)
    }
    
    // MARK: - Trend Analysis Tests
    
    func testCalculateTrend() async throws {
        let baseDate = Date()
        let predictions = [
            PredictionResult(date: Calendar.current.date(byAdding: .day, value: -30, to: baseDate)!, lifeExpectancy: 75.0, riskFactors: [:], confidence: 0.8),
            PredictionResult(date: Calendar.current.date(byAdding: .day, value: -20, to: baseDate)!, lifeExpectancy: 76.0, riskFactors: [:], confidence: 0.8),
            PredictionResult(date: Calendar.current.date(byAdding: .day, value: -10, to: baseDate)!, lifeExpectancy: 77.5, riskFactors: [:], confidence: 0.8),
            PredictionResult(date: baseDate, lifeExpectancy: 78.0, riskFactors: [:], confidence: 0.8)
        ]
        
        for prediction in predictions {
            await databaseService.savePrediction(prediction)
        }
        
        let trend = await predictionHistoryManager.calculateTrend(days: 30)
        XCTAssertGreaterThan(trend.slope, 0) // Should be positive trend
        XCTAssertGreaterThan(trend.correlation, 0.5) // Should be reasonably correlated
    }
    
    func testGetAverageLifeExpectancy() async throws {
        let predictions = [
            PredictionResult(date: Date(), lifeExpectancy: 75.0, riskFactors: [:], confidence: 0.8),
            PredictionResult(date: Date(), lifeExpectancy: 77.0, riskFactors: [:], confidence: 0.8),
            PredictionResult(date: Date(), lifeExpectancy: 79.0, riskFactors: [:], confidence: 0.8)
        ]
        
        for prediction in predictions {
            await databaseService.savePrediction(prediction)
        }
        
        let average = await predictionHistoryManager.getAverageLifeExpectancy(days: 30)
        XCTAssertEqual(average, 77.0, accuracy: 0.01)
    }
    
    // MARK: - Performance Tests
    
    func testBatchInsertPerformance() throws {
        let predictions = (0..<1000).map { i in
            PredictionResult(
                date: Calendar.current.date(byAdding: .day, value: -i, to: Date())!,
                lifeExpectancy: Double(75 + i % 10),
                riskFactors: ["test": Double(i % 100) / 100.0],
                confidence: 0.8
            )
        }
        
        measure {
            Task {
                for prediction in predictions {
                    await databaseService.savePrediction(prediction)
                }
            }
        }
    }
    
    func testQueryPerformance() async throws {
        // Insert test data
        let predictions = (0..<100).map { i in
            PredictionResult(
                date: Calendar.current.date(byAdding: .day, value: -i, to: Date())!,
                lifeExpectancy: Double(75 + i % 10),
                riskFactors: [:],
                confidence: 0.8
            )
        }
        
        for prediction in predictions {
            await databaseService.savePrediction(prediction)
        }
        
        measure {
            Task {
                _ = await databaseService.getAllPredictions()
            }
        }
    }
    
    // MARK: - Edge Cases
    
    func testEmptyDatabase() async throws {
        let predictions = await databaseService.getAllPredictions()
        XCTAssertEqual(predictions.count, 0)
        
        let average = await predictionHistoryManager.getAverageLifeExpectancy(days: 30)
        XCTAssertEqual(average, 0)
    }
    
    func testDuplicateEntries() async throws {
        let date = Date()
        let prediction1 = PredictionResult(date: date, lifeExpectancy: 75.0, riskFactors: [:], confidence: 0.8)
        let prediction2 = PredictionResult(date: date, lifeExpectancy: 78.0, riskFactors: [:], confidence: 0.8)
        
        await databaseService.savePrediction(prediction1)
        await databaseService.savePrediction(prediction2)
        
        let predictions = await databaseService.getAllPredictions()
        XCTAssertEqual(predictions.count, 2) // Both should be saved
    }
    
    func testExtremeValues() async throws {
        let extremePredictions = [
            PredictionResult(date: Date(), lifeExpectancy: 0.0, riskFactors: [:], confidence: 0.0),
            PredictionResult(date: Date(), lifeExpectancy: 150.0, riskFactors: [:], confidence: 1.0),
            PredictionResult(date: Date(), lifeExpectancy: -10.0, riskFactors: [:], confidence: -0.5)
        ]
        
        for prediction in extremePredictions {
            await databaseService.savePrediction(prediction)
        }
        
        let savedPredictions = await databaseService.getAllPredictions()
        XCTAssertEqual(savedPredictions.count, 3)
        
        let average = await predictionHistoryManager.getAverageLifeExpectancy(days: 30)
        XCTAssertGreaterThanOrEqual(average, 0) // Should handle negative values gracefully
    }
}